/**
* BaseCore.ts
* Base Core is a subset of 1DS Web SDK Core. The purpose of Base Core is to generate a smaller bundle size while providing essential features of Core. Features that are not included in Base Core are:
* 1. Internal logging
* 2. Sending notifications on telemetry sent/discarded
* @author Abhilash Panwar (abpanwar) Hector Hernandez (hectorh)
* @copyright Microsoft 2018
*/
import dynamicProto from "@microsoft/dynamicproto-js";
import {
    BaseCore as InternalCore, IDiagnosticLogger, INotificationManager, IPlugin, ITelemetryItem, _throwInternal, dumpObj, eLoggingSeverity
} from "@microsoft/applicationinsights-core-js";
import { IExtendedAppInsightsCore, IExtendedConfiguration, IExtendedTelemetryItem } from "./DataModels";
import { _eExtendedInternalMessageId } from "./Enums";
import { STR_DEFAULT_ENDPOINT_URL } from "./InternalConstants";
import { FullVersionString, isDocumentObjectAvailable } from "./Utils";

export default class BaseCore extends InternalCore implements IExtendedAppInsightsCore {

    // Function to get w parameter used by OneCollector
    public getWParam: () => number;

    constructor() {
        super();

        dynamicProto(BaseCore, this, (_self, _base) => {

            _self.initialize = (config: IExtendedConfiguration, extensions: IPlugin[], logger?: IDiagnosticLogger, notificationManager?: INotificationManager) => {
                if (config && !config.endpointUrl) {
                    config.endpointUrl = STR_DEFAULT_ENDPOINT_URL;
                }

                _self.getWParam = () => {
                    return (isDocumentObjectAvailable || !!config.enableWParam) ? 0 : -1;
                };

                try {
                    _base.initialize(config, extensions, logger, notificationManager);
                } catch (e) {
                    _throwInternal(_self.logger,
                        eLoggingSeverity.CRITICAL,
                        _eExtendedInternalMessageId.FailedToInitializeSDK, "Initialization Failed: " + dumpObj(e) + "\n - Note: Channels must be provided through config.channels only"
                    );
                }
            };

            _self.track = (item: IExtendedTelemetryItem|ITelemetryItem) => {
                let telemetryItem: IExtendedTelemetryItem = item as IExtendedTelemetryItem;
                if (telemetryItem) {
                    let ext = telemetryItem.ext = telemetryItem.ext || {};
                    ext.sdk = ext.sdk || {};
                    ext.sdk.ver = FullVersionString;
                }

                _base.track(telemetryItem);
            };
        });
    }

    /**
     * Initialize the sdk.
     * @param config - The configuration to initialize the SDK.
     * @param extensions - An array of extensions that are to be used by the core.
     */
    public initialize(config: IExtendedConfiguration, extensions: IPlugin[], logger?: IDiagnosticLogger, notificationManager?: INotificationManager) {
        // @DynamicProtoStub -- DO NOT add any code as this will be removed during packaging
    }

    public track(item: IExtendedTelemetryItem|ITelemetryItem) {
        // @DynamicProtoStub -- DO NOT add any code as this will be removed during packaging
    }
}
